<?php

if ( ! defined( 'ABSPATH' ) ) {
    die( '-1' );
}

if ( !class_exists( 'SocialFans_Counter_Auto_Update' ) )
{

    class SocialFans_Counter_Auto_Update
    {
        /**
         * The plugin current version
         * @var string
         */
        public $current_version;

        /**
         * The plugin remote update path
         * @var string
         */
        public $update_path;

        /**
         * Plugin Slug (plugin_directory/plugin_file.php)
         * @var string
         */
        public $plugin_slug;

        /**
         * Plugin name (plugin_file)
         * @var string
         */
        public $slug;

        /**
         * Initialize a new instance of the WordPress Auto-Update class
         * @param string $current_version
         * @param string $update_path
         * @param string $plugin_slug
         */
        function __construct($current_version, $update_path, $plugin_slug, $plugin_item_id)
        {
            #var_dump( wp_debug_backtrace_summary() );
            // Set the class public variables
            $this->current_version = $current_version;
            $this->update_path     = $update_path;
            $this->plugin_slug     = $plugin_slug;
            $this->plugin_id       = $plugin_item_id;
            list ($t1, $t2) = explode('/', $plugin_slug);
            $this->slug = str_replace('.php', '', $t2);

            $this->init();
        }

        public function init()
        {
            // define the alternative API for updating checking
            add_filter('pre_set_site_transient_update_plugins', array (
                $this,
                'check_update'
            ));

            // Define the alternative response for information checking
            add_filter('plugins_api', array (
                $this,
                'check_info'
            ), 10, 3);
            add_filter('plugins_api_result', array (
                $this,
                'get_info'
            ), 10, 2);

            add_filter('upgrader_pre_download', array (
                $this,
                'preUpgradeFilter'
            ), 10, 4);
            add_filter('http_request_host_is_external', '__return_true');
        }

        /**
         * Add our self-hosted autoupdate plugin to the filter transient
         *
         * @param $transient
         * @return object $ transient
         */
        public function check_update($transient)
        {
            if (empty($transient->checked))
            {
                return $transient;
            }

            // Get the remote version
            $remote_version = $this->getRemote_version();

            // If a newer version is available, add the update
            if (version_compare($this->current_version, $remote_version, '<'))
            {
                $obj                                     = new stdClass();
                $obj->plugin                             = 'sf-counter';
                $obj->slug                               = $this->slug;
                $obj->new_version                        = $remote_version;
                $obj->url                                = $this->update_path;
                $obj->package                            = $this->update_path;
                $transient->response[$this->plugin_slug] = $obj;
            }

            return $transient;
        }

        /**
         * Add our self-hosted description to the filter
         *
         * @param boolean $false
         * @param array $action
         * @param object $arg
         * @return bool|object
         */
        public function check_info($false, $action, $arg)
        {
            if ($arg->slug === $this->slug)
            {
                $information = $this->getRemote_information();
                return $information;
            }

            return false;
        }

        public function get_info($arg)
        {
            if ($arg->slug === $this->slug)
            {
                $information           = $this->getRemote_information();
                $information->sections = json_decode(json_encode($information->sections), true);
                return $information;
            }
            return $arg;
        }

        /**
         * Return the remote version
         * @return string $remote_version
         */
        public function getRemote_version()
        {
            $request = wp_remote_get($this->update_path . '/version?item=' . $this->plugin_id);

            if (!is_wp_error($request) || wp_remote_retrieve_response_code($request) === 200)
            {
                return $request['body'];
            }

            return false;
        }

        /**
         * Get information about the remote version
         * @return bool|object
         */
        public function getRemote_information()
        {
            $request = wp_remote_get($this->update_path . '/info?item=' . $this->plugin_id);

            if (!is_wp_error($request) || wp_remote_retrieve_response_code($request) === 200)
            {
                return json_decode($request['body']);
            }

            return false;
        }

        /**
         * Return the status of the plugin licensing
         * @return boolean $remote_license
         */
        public function getRemote_license()
        {

            $request = wp_remote_post('kamal', array ('body' => array ('action' => 'license')));

            if (!is_wp_error($request) || wp_remote_retrieve_response_code($request) === 200)
            {
                return $request['body'];
            }

            return false;
        }

        protected function getUrl()
        {
            return $this->update_path . '/download-link?item=' . $this->plugin_id . '&key=' . rawurlencode(SocialFans_License::getPurchaseCode()) . '&version=' . SOCIALFANS_COUNTER_VERSION . '&url=' . esc_url(get_site_url());
        }

        public function getDownloadUrl()
        {
            $url = $this->getUrl();

            $response = wp_remote_get($url);

            if (is_wp_error($response))
            {
                return false;
            }

            return json_decode($response['body'], true);
        }

        public function preUpgradeFilter($reply, $package, $updater)
        {
            if ( $updater->skin->plugin !=  $this->plugin_slug)
                return $reply;

            $res = $updater->fs_connect(array (WP_CONTENT_DIR));

            if (!$res)
                return new WP_Error('no_credentials', __("Error! Can't connect to filesystem", 'sfcounter'));


            if (!ShareBang_License::isPluginActiveted())
                return new WP_Error('no_credentials', __("To receive automatic updates license activation is required", 'sfcounter'));

            $updater->strings['downloading_package_url'] = __('Getting download link...', 'sfcounter');
            $updater->skin->feedback('downloading_package_url');

            $response = $this->getDownloadUrl();

            if (!$response)
            {
                return new WP_Error('no_credentials', __('Download link could not be retrieved', 'sfcounter'));
            }


            if (!$response['status'])
            {
                return new WP_Error('no_credentials', $response['error']);
            }

            $updater->strings['downloading_package'] = __('Downloading package...', 'sfcounter');
            $updater->skin->feedback('downloading_package');

            $downloaded_archive = download_url($response['url']);

            if (is_wp_error($downloaded_archive))
            {
                return $downloaded_archive;
            }

            return $downloaded_archive;

        }
    }
}